# Script para identificar quando a correlação NCM-Naladi deve ser feita 
# manualmente.

# No caso em que uma mesma NCM está relacionada com mais de um código NALADI,
# isto é, quando tem um fechamento Naladi para NCM de N para 1 e, quando essa
# mesma NCM tem um desdobramento de 1 para N. É necessário fazer a correlação
# manualmente

# OBS. Rodar este script antes de gera_correlacoes.R

# OBS2. Rodar script linha a linha e analisar


library(tidyverse)
library(readxl)


# ver ncms criadas
inclusao <- read_xlsx(
  file.path(
    Sys.getenv("general"),
    "Bases",
    "correlacoes",
    "mercosul",
    "input",
    "alteracao_manual",
    "inclusao.xlsx"
))


# carrega ncm_naladi manual
path_naladi_ncm_manual <- file.path(
  Sys.getenv("general"),
  "Bases",
  "correlacoes",
  "aladi",
  "naladi_ncm",
  "correlacao_ncm_naladi_manual.xlsx"
)

naladi_ncm_manual <- read_xlsx(
  path_naladi_ncm_manual
  ) %>% 
  mutate(
    across(
      everything(),
      ~str_remove_all(.x, "\\.")
    )
  ) %>% 
  rename_with(
    ~ paste0("naladi"),
    contains("naladi")
  ) %>% 
  rename_with(
    ~ paste0("ncm"),
    contains("ncm")
  )


# ver dados resolucoes gecex
inclusao %>% 
  count(res)

# filtrar última resolução
ultima_res <- inclusao %>% 
  filter(res == 812)

# obter ncms antigas que se desdobraram
ncms_antigas_desdobramento <- ultima_res %>% 
  count(ncm_antiga) %>% 
  filter(n > 1) %>% 
  pull(ncm_antiga)

# obs. somente as que se desdobraram é que podem ter problemas de classificação
# entre naladi e ncm. O problema surge quando 2 ou mais naladi se relacionam com um
# único código ncm antigo que se desdobra em vários novos.


# se houver naladis diferentes que fecham em um código ncm, precisa fazer
# correlação manual

# conjunto de ncms que precisam ser definidas manualmente na correlação
verifica_manual <- naladi_ncm_manual %>% 
  filter(ncm %in% ncms_antigas_desdobramento) %>% 
  count(ncm) %>% 
  filter(n > 1) %>% 
  pull(ncm)

if(!is_empty(verifica_manual)) {
  message("Elaborar correlação manualmente!!!\n")
  message("Olhar descrições NALADI e NCM (imprensa nacional se ainda não foi publicado) e alterar diretamente no arquivo:\n")
  message(path_naladi_ncm_manual)
  naladi_ncm_manual %>%
    filter(ncm %in% verifica_manual) %>%
    # join para obter ncms novas
    left_join(
      ultima_res,
      by = c("ncm" = "ncm_antiga"),
      relationship = "many-to-many"
    ) %>% 
    select(-ncm, -res) %>% 
    print()
  
  stop("Apos alterar no arquivo excel rodar script de novo para atualizar dados")
  
} 

# identificar e remover o desdobramento que não existiu.
# exemplo:
# naladi   ncm_nova
# 29159010 29159070 # o código naladi está em outros (final 90) e não no final 70. Descartar
# 29159010 29159090 # essa correlação ja existe, esta ok
# 29159090 29159070 # aqui vemos que esse codigo surgiu do outro (final 90)
# 29159090 29159090 # essa correlação já existia

# isto é, eliminar naladi 29159010 / ncm_nova 29159070 pois correlação não procede


temp <- ultima_res %>% 
  filter(ncm_antiga %in% ncms_antigas_desdobramento) %>%
  distinct() %>% 
  # obter códigos naladi correspondentes para ncms novas
  left_join(
    naladi_ncm_manual,
    by = c("ncm_antiga" = "ncm"),
    relationship = 'many-to-many'
  ) %>% 
  # eliminar correlações inexistentes, se houver
  # caso não haja, comentar linha de baixo
  filter(!(ncm_nova == "29159070" & naladi == "29159010"))
  

ncms_mantidas <- temp %>% 
  filter(ncm_antiga == ncm_nova) %>% 
  pull(ncm_antiga) %>% 
  unique()

# selecao de ncms para remover, manter apenas as que foram mantidas entre
# antiga e nova
ncms_para_remover <- temp %>% 
  filter(!(ncm_antiga %in% ncms_mantidas)) %>% 
  pull(ncm_antiga) %>% 
  unique()

# obtem correlacao naladi com ncms novas desdobradas
atualizacao_ncm_naladi <- temp %>% 
  filter(!is.na(naladi)) %>% 
  select(naladi, ncm_nova) %>% 
  distinct() %>% 
  rename_with(
    ~ paste0("ncm"),
    contains("ncm")
  )

# corrige codigo ncm sem correlacao naladi
atualizacao_ncm_naladi_na <- temp %>% 
  filter(is.na(naladi)) %>% 
  select(-naladi) %>% 
  left_join(naladi_ncm_manual, by = c("ncm_antiga" = "ncm")) %>% 
  select(naladi, ncm_nova) %>% 
  rename_with(
    ~ paste0("ncm"),
    contains("ncm")
  )

# testa se há dados vazios em naladi
teste <- atualizacao_ncm_naladi_na %>% 
  filter(is.na(naladi))

if(nrow(teste) > 0) {
  cat("Veja Naladi sem correspondência")
  print(teste)
  stop("Naladi sem correspondência para nova NCM\n")
  
}

atualizacao_final <- bind_rows(
  atualizacao_ncm_naladi,
  atualizacao_ncm_naladi_na,
  naladi_ncm_manual
) %>% 
  arrange(naladi) %>% 
  distinct() %>% 
  # remove ncms que deixaram de existir na correlação
  filter(!(ncm %in% ncms_para_remover)) %>% 
  mutate(
    across(
      everything(),
      ~ paste0(
        str_sub(.x, 1, 4), ".",
        str_sub(.x, 5, 6), ".",
        str_sub(.x, 7, 8)
      )
    )
  )

# abre dados só para obter nomes das colunas originais
dados_naladi_ncm_manual <- read_xlsx(
  path_naladi_ncm_manual
)

nomes <- names(dados_naladi_ncm_manual)

novo_naladi <- nomes[str_detect(nomes, regex("^NALADI(_|$)", ignore_case = TRUE))][1]
novo_ncm <- nomes[str_detect(nomes, regex("^NCM(_|$)",    ignore_case = TRUE))][1]

# coloca nomes das colunas originais
atualizacao_final <- atualizacao_final %>%
  rename(
    !!novo_naladi := naladi,
    !!novo_ncm    := ncm
  )

# salva atualização de correlação naladi ncm
writexl::write_xlsx(
  atualizacao_final,
  # para sobreescrever arquivo original
  path_naladi_ncm_manual
  
  
)



