library(dplyr)
library(tidyselect)
library(stringr)
library(nanoparquet)
library(tidyr)
library(readxl)
library(purrr)

# carrega caminhos para dados

path_ncm <- file.path(
  Sys.getenv("general"),
  "Bases",
  "correlacoes",
  "mercosul",
  "output",
  "correlacao_completa_ncm.Rda"
)

path_ncm_detalhada <- file.path(
  Sys.getenv("general"),
  "Bases",
  "correlacoes",
  "mercosul",
  "output",
  "correlacao_completa_ncm_detalhada.xlsx"
)

path_naladi_ncm <- file.path(
  Sys.getenv("general"),
  "Bases",
  "correlacoes",
  "aladi",
  "naladi_ncm",
  "output",
  "correlacao_naladi_ncm.Rda"
)

path_naladi_ncm_manual <- file.path(
  Sys.getenv("general"),
  "Bases",
  "correlacoes",
  "aladi",
  "naladi_ncm",
  # "correlacao_ncm_naladi_manual.xlsx"
  "correlacao_ncm_naladi_res_812.xlsx"
)

path_naladi <- file.path(
  Sys.getenv("general"),
  "Bases",
  "correlacoes",
  "aladi",
  "naladi_naladi",
  "output",
  "correlacao_completa_naladi_ncmtracker.Rda"
)

path_descricoes <- file.path(
  Sys.getenv("general"),
  "Bases",
  "NCM_sh1996_a_sh2022",
  "raw",
  "tabela_nomenclatura.parquet"
) 


# Obter códigos que mudaram nas revisões ----------------------------------

correlacao_ncm_detalhada <- read_xlsx(path_ncm_detalhada, guess_max = 1e6)

# funcao para obter linhas com mudancas de codigo em cada revisao NCM
filtra_mudancas <- function(x, padrao) {
  x |> 
    select(matches(padrao)) |> 
    filter(
      # seleciona as linhas que tem algum valor diferente de NA com exceção
      # de dados na primeira coluna
      if_any(-1, ~ !is.na(.))
    ) |> 
    # cria uma coluna preenchendo-a com a primeira informação não NA da
    # direita para a esquerda
    mutate(!!padrao := do.call(coalesce, rev(as.list(across(-1))))) |> 
    select(
      ncm_original = 1,
      last_col()
    ) |> 
    distinct()
}

# cria lista de nomes para usar como input em filtra_mudancas
nomes_para_funcao <- names(correlacao_ncm_detalhada) |> 
  str_extract("NCM_\\d{4}") |> 
  unique()

# obtem lista com as ncms originais das revisoes
ncms_originais <- nomes_para_funcao |> 
  map(~ filtra_mudancas(correlacao_ncm_detalhada, padrao = .x))

names(ncms_originais) <- nomes_para_funcao |>
  str_replace("_", " ")

saveRDS(ncms_originais, "dados_gerados/ncms_originais.rds")

# Organiza dados da correlação NCM ----------------------------------------


load(path_ncm)

correlacao_completa <- correlacao_completa_ncm_mercosul |> 
  distinct() |> 
  rename_with(~str_to_upper(.x))

write_parquet(correlacao_completa, "dados_gerados/dataset_ncm.parquet")


# Organiza dados da correlação NCM Naladi ---------------------------------

# load(path_naladi_ncm)

naladi_ncm_manual <- read_xlsx(
  path_naladi_ncm_manual,
  guess_max = 1e6)

colunas <- names(naladi_ncm_manual)

ano_max <- colunas |> 
  str_extract("\\d{4}") |> 
  max(na.rm = TRUE) |> 
  unique()

naladi_ncm_manual <- naladi_ncm_manual |> 
  select(contains(ano_max)) |> 
  distinct()


naladi_ncm_manual |> 
  write_parquet("dados_gerados/dataset_naladi_ncm.parquet")


# Organiza dados da correlação Naladi -------------------------------------

load(path_naladi)

correlacao_completa <- correlacao_completa_naladi_naladi_ncmtracker |> 
  # naladi 1983 tem 7 ou 8 digitos. os 8 digitos da naladi 1983 surgiram em 1992.
  # como logo em seguida veio a naladi_1993 com 8 digitos, vamos
  # descartar os codigos de 8 dígitos da naladi_1983.
  # filter(nchar(naladi_1983) == 9) |>  # 9 digitos equivale a 7 digitos se contarmos os pontos
  distinct() |> 
  rename_with(~str_to_upper(.x)) |> 
  mutate(NALADI_1983 = str_remove_all(NALADI_1983, "\\.")) |> 
  mutate(NALADI_1983 = str_c(
    str_sub(NALADI_1983, 1, 2),
    ".",
    str_sub(NALADI_1983, 3, 4),
    ".",
    str_sub(NALADI_1983, 5, 5),
    ".",
    str_sub(NALADI_1983, 6, 7)
  ))

colunas <- names(correlacao_completa)

write_parquet(correlacao_completa, "dados_gerados/dataset_naladi.parquet")


# Processa e organiza descrições NCM --------------------------------------
cria_padrao <- function(x) {
  paste0(x, "$", collapse = "|")
}


descricoes_raw <- arrow::read_parquet(path_descricoes) |> 
  select(
    codigo,
    nomeExtenso,
    inicioVigencia,
    inicioVigenciaStr,
    fimVigencia,
    fimVigenciaStr
    ) |> 
  mutate(
    grupo_sh = case_when(
      str_detect(fimVigenciaStr, cria_padrao(1996:2001)) ~ "1996",
      str_detect(fimVigenciaStr, cria_padrao(2002:2006)) ~ "2002",
      str_detect(fimVigenciaStr, cria_padrao(2007:2011)) ~ "2007",
      str_detect(fimVigenciaStr, cria_padrao(2012:2016)) ~ "2012",
      fimVigencia == 1648695600000 ~ "2017",
      TRUE ~ "2022",
    )
  ) |> 
  group_by(grupo_sh, codigo) |> 
  slice_max(order_by = fimVigencia, n = 1) |> 
  ungroup()


x <- descricoes_raw |> 
  filter(nchar(codigo) != 2) |> 
  select(codigo, nomeExtenso, grupo_sh) |> 
  mutate(
    n = nchar(codigo)
  )

sh4 <- x |> 
  filter(n == 4) |> 
  select(-n) |>  
  rename(sh4 = codigo) |> 
  rename(desc_sh4 = nomeExtenso) |> 
  mutate(desc_sh4 = str_squish(desc_sh4))

sh5 <- x |> 
  filter(
    n == 5
  ) |> 
  select(-n) |> 
  rename(sh5 = codigo) |> 
  rename(desc_sh5 = nomeExtenso) |> 
  mutate(desc_sh5 = str_squish(desc_sh5))
  
sh6 <- x |> 
  filter(n == 6) |> 
  select(-n) |>  
  rename(sh6 = codigo) |> 
  rename(desc_sh6 = nomeExtenso) |> 
  mutate(desc_sh6 = str_squish(desc_sh6))

sh7 <- x |> 
  filter(n == 7) |> 
  select(-n) |>  
  rename(sh7 = codigo) |> 
  rename(desc_sh7 = nomeExtenso) |> 
  mutate(desc_sh7 = str_squish(desc_sh7))

descricoes_por_sh <- x |> 
  filter(n == 8) |> 
  select(-n) |> 
  mutate(
    sh4 = str_sub(codigo, 1, 4),
    sh5 = str_sub(codigo, 1, 5),
    sh6 = str_sub(codigo, 1, 6),
    sh7 = str_sub(codigo, 1, 7),
    nomeExtenso = str_replace(
      nomeExtenso,
      "^- ",
      "-"
    )
  ) |> 
  mutate(
    nomeExtenso = str_replace(
      nomeExtenso,
      "^-- ",
      "--"
    )
  ) |> 
  left_join(sh4, by = c("sh4", "grupo_sh")) |> 
  left_join(sh5, by = c("sh5", "grupo_sh")) |> 
  left_join(sh6, by = c("sh6", "grupo_sh")) |> 
  left_join(sh7, by = c("sh7", "grupo_sh")) |> 
  select(-matches("^sh")) |> 
  mutate(
    across(
      starts_with("desc_sh"),
      ~str_replace(.x, "^- ", "-")
    ),
    across(
      starts_with("desc_sh"),
      ~str_replace(.x, "^-- ", "--")
    )
  ) |> 
  unite(
    col = descricao,
    starts_with("desc_sh"), nomeExtenso,
    sep = "\n",
    na.rm = TRUE
  ) |> 
  select(
    ncm = codigo,
    sh = grupo_sh,
    descricao
  ) |> 
  mutate(
    sh = paste("NCM", sh),
    ncm = paste0(
      str_sub(ncm, 1, 4), ".",
      str_sub(ncm, 5, 6), ".",
      str_sub(ncm, 7, 8)
    ))


# necessário escapar ' para não dar problema na geração dos tooltips dos
# diagramas
descricoes_por_sh <- descricoes_por_sh %>% 
  mutate(
    descricao = str_replace_all(
      descricao,
      "'",
      "\\\\'"
    )) %>% 
  mutate(
    descricao = str_replace_all(
      descricao,
      "\\r\\n",
      " "
    ),
    descricao = str_trunc(
      descricao,
      1020,
      'left'
    )
  )

descricoes_por_sh %>% 
  arrow::write_parquet(
    "dados_gerados/descricoes_ncm.parquet"
  )
