from flask import Flask, render_template, request, send_file
from rdflib import Graph, Namespace, URIRef, Literal
from rdflib.namespace import DCTERMS, FOAF, RDF, XSD  # Importação corrigida
import uuid


app = Flask(__name__)

# Namespaces
DCAT = Namespace("http://www.w3.org/ns/dcat#")
ADMS = Namespace("http://www.w3.org/ns/adms#")
VCARD = Namespace("http://www.w3.org/2006/vcard/ns#")

@app.route('/')
def form():
    return render_template('form.html')

@app.route('/generate_rdf', methods=['POST'])
def generate_rdf():
    # Validar campos obrigatórios
    required_fields = ['title', 'description', 'publisher', 'contactPoint', 'license', 'accrualPeriodicity']
    for field in required_fields:
        if not request.form.get(field):
            return f"Campo obrigatório faltando: {field}", 400

    # Criar grafo RDF
    g = Graph()
    g.bind("dcterms", DCTERMS)
    g.bind("dcat", DCAT)
    g.bind("foaf", FOAF)
    g.bind("vcard", VCARD)
    g.bind("adms", ADMS)

    # URIs
    dataset_id = URIRef(f"https://dados.gov.br/dados/conjuntos-dados/{uuid.uuid4()}")
    creator_id = URIRef(f"https://dados.gov.br/dados/organizacoes/{uuid.uuid4()}")
    contact_point_id = URIRef(f"mailto:{request.form['contactPoint']}")

    # Dataset


    g.add((dataset_id, RDF.type, DCAT.Dataset))
    g.add((dataset_id, DCTERMS.title, Literal(request.form['title'])))  # Agora funcionará
    g.add((dataset_id, DCTERMS.description, Literal(request.form['description'])))
    g.add((dataset_id, DCTERMS.publisher, Literal(request.form['publisher'])))
    g.add((dataset_id, DCAT.license, URIRef(request.form['license'])))
    g.add((dataset_id, DCTERMS.accrualPeriodicity, Literal(request.form['accrualPeriodicity'])))
    g.add((dataset_id, DCTERMS.accessRights, Literal(request.form['accessRights'])))
    g.add((dataset_id, ADMS.version, Literal(request.form['version'])))

    # Criador (Organização)
    g.add((dataset_id, DCTERMS.creator, creator_id))
    g.add((creator_id, RDF.type, FOAF.Organization))
    g.add((creator_id, FOAF.name, Literal(request.form['creator'])))

    # Contact Point
    g.add((dataset_id, DCAT.contactPoint, contact_point_id))
    g.add((contact_point_id, RDF.type, VCARD.Organization))
    g.add((contact_point_id, VCARD.hasEmail, contact_point_id))

    # Palavras chave
    for keyword in request.form['keywords'].split(','):
        g.add((dataset_id, DCAT.keyword, Literal(keyword.strip())))

    # Temas
    for theme in request.form['theme'].split(','):
        g.add((dataset_id, DCAT.theme, Literal(theme.strip())))

    # Salvar RDF em arquivo temporário
    rdf_content = g.serialize(format='pretty-xml')
    
    return rdf_content, 200, {
        'Content-Type': 'application/rdf+xml',
        'Content-Disposition': 'attachment; filename=metadata.rdf'
    }

if __name__ == '__main__':
    app.run(debug=True)